<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Spatie\Activitylog\LogOptions;
use Spatie\Activitylog\Traits\LogsActivity;

class Configuration extends Model
{
    use LogsActivity;

    protected $fillable = [
        'key',
        'value',
        'type',
        'description',
        'group',
        'is_public'
    ];

    protected $casts = [
        'is_public' => 'boolean',
    ];

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logAll()
            ->logOnlyDirty()
            ->dontSubmitEmptyLogs()
            ->setDescriptionForEvent(fn(string $eventName) => "Configuration has been {$eventName}");
    }

    /**
     * Récupère une valeur de configuration par sa clé
     */
    public static function get($key, $default = null)
    {
        $config = static::where('key', $key)->first();
        
        if (!$config) {
            return $default;
        }

        return static::castValue($config->value, $config->type);
    }

    /**
     * Définit une valeur de configuration
     */
    public static function set($key, $value, $type = 'string', $description = null, $group = 'general', $isPublic = false)
    {
        return static::updateOrCreate(
            ['key' => $key],
            [
                'value' => $value,
                'type' => $type,
                'description' => $description,
                'group' => $group,
                'is_public' => $isPublic
            ]
        );
    }

    /**
     * Cast la valeur selon le type
     */
    private static function castValue($value, $type)
    {
        switch ($type) {
            case 'boolean':
                return filter_var($value, FILTER_VALIDATE_BOOLEAN);
            case 'integer':
                return (int) $value;
            case 'json':
                return json_decode($value, true);
            default:
                return $value;
        }
    }

    /**
     * Récupère toutes les configurations d'un groupe
     */
    public static function getGroup($group)
    {
        return static::where('group', $group)->get()->mapWithKeys(function ($config) {
            return [$config->key => static::castValue($config->value, $config->type)];
        });
    }

    /**
     * Récupère toutes les configurations publiques
     */
    public static function getPublic()
    {
        return static::where('is_public', true)->get()->mapWithKeys(function ($config) {
            return [$config->key => static::castValue($config->value, $config->type)];
        });
    }
}
